////////////////////////////////////////////////////////////////////////////////
// SoftKinetic DepthSense SDK
//
// COPYRIGHT AND CONFIDENTIALITY NOTICE - SOFTKINETIC CONFIDENTIAL
// INFORMATION
//
// All rights reserved to SOFTKINETIC SENSORS NV (a
// company incorporated and existing under the laws of Belgium, with
// its principal place of business at Boulevard de la Plainelaan 15,
// 1050 Brussels (Belgium), registered with the Crossroads bank for
// enterprises under company number 0811 341 454 - "Softkinetic
// Sensors").
//
// The source code of the SoftKinetic DepthSense Camera Drivers is
// proprietary and confidential information of Softkinetic Sensors NV.
//
// For any question about terms and conditions, please contact:
// info@softkinetic.com Copyright (c) 2002-2012 Softkinetic Sensors NV
////////////////////////////////////////////////////////////////////////////////


#ifdef _MSC_VER
#include <windows.h>
#endif

#include <stdio.h>
#include <vector>
#include <exception>

#include <DepthSense.hxx>

using namespace DepthSense;
using namespace std;

/* OpenCV */
#include "opencv2\\opencv.hpp"
// debug
#pragma comment(lib, "opencv_core240d.lib")
#pragma comment(lib, "opencv_highgui240d.lib")
#pragma comment(lib, "opencv_imgproc240d.lib")
// release
//#pragma comment(lib, "opencv_core240.lib")
//#pragma comment(lib, "opencv_highgui240.lib")
//#pragma comment(lib, "opencv_imgproc240.lib")
using namespace cv;

#define WINDOW_DEPTH	"SoftKinetic Depth Camera"
#define WINDOW_COLOR	"SoftKinetic Color Camera"
#define DEPTH_WIDTH		320
#define DEPTH_HEIGHT	240
#define COLOR_WIDTH		640
#define COLOR_HEIGHT	480

Context g_context;
DepthNode g_dnode;
ColorNode g_cnode;
AudioNode g_anode;

uint32_t g_aFrames = 0;
uint32_t g_cFrames = 0;
uint32_t g_dFrames = 0;

bool g_bDeviceFound = false;

ProjectionHelper* g_pProjHelper = NULL;
StereoCameraParameters g_scp;

/*----------------------------------------------------------------------------*/
// New audio sample event handler
void onNewAudioSample(AudioNode node, AudioNode::NewSampleReceivedData data)
{
    // Do Nothing
}

/*----------------------------------------------------------------------------*/
// New color sample event handler
void onNewColorSample(ColorNode node, ColorNode::NewSampleReceivedData data)
{
    int memsize = COLOR_WIDTH * COLOR_HEIGHT * 3 * sizeof(char); // RGB  摜obt@TCY
	unsigned char *rgbimage = (unsigned char *)malloc(memsize);
	for(int i=0; i<memsize; i++) rgbimage[i] = 0;

	Mat mat( COLOR_HEIGHT, COLOR_WIDTH, CV_8UC3, (void*)(const uint8_t*)data.colorMap );
	
	/*------ start of YUY2 -> RGB ------*/
	int i, j;
	int Y0, V0, U0, Y1;
	int R, G, B;
	int img_addr;
	unsigned int nCount = 0;

	for (i=0; i<(COLOR_HEIGHT/2); i++){ for (j=0; j<COLOR_WIDTH; j++)
	{
		img_addr = (i * COLOR_WIDTH + j) * 4;
		Y0 = mat.data[img_addr  ] & 0xff;
		V0 = mat.data[img_addr+1] & 0xff; /* Cr F() */
		Y1 = mat.data[img_addr+2] & 0xff;
		U0 = mat.data[img_addr+3] & 0xff; /* Cb F() */

		// V0, U0 A2 dot ɂA1 ̏ƂȂ
		R = (int)(1.164 * (Y0-16) + 1.567 * (V0-128));
		G = (int)(1.164 * (Y0-16) - 0.798 * (V0-128) - 0.384 * (U0-128));
		B = (int)(1.164 * (Y0-16) + 1.980 * (U0-128));

		if (R < 0) R= 0; if (R > 255) R= 255;
		if (G < 0) G= 0; if (G > 255) G= 255;
		if (B < 0) B= 0; if (B > 255) B= 255;

		// R, G, B obt@[ɏ
		rgbimage[nCount++] = R;
		rgbimage[nCount++] = G;
		rgbimage[nCount++] = B;
		
		R = (int)(1.164 * (Y1-16) + 1.567 * (V0-128));
		G = (int)(1.164 * (Y1-16) - 0.798 * (V0-128) - 0.384 * (U0-128));
		B = (int)(1.164 * (Y1-16) + 1.980 * (U0-128));

		if (R < 0) R= 0;
		if (R > 255) R= 255;

		if (G < 0) G= 0;
		if (G > 255) G= 255;

		if (B < 0) B= 0;
		if (B > 255) B= 255;

		// R, G, B obt@[ɏ
		rgbimage[nCount++] = R;
		rgbimage[nCount++] = G;
		rgbimage[nCount++] = B;
	}}
	/*------ end of YUY2 -> RGB ------*/
	
	IplImage *colorImage8U = cvCreateImage(cvSize(COLOR_WIDTH, COLOR_HEIGHT), IPL_DEPTH_8U, 3); // 摜obt@
	memcpy(colorImage8U->imageData, rgbimage, memsize);
	cvShowImage(WINDOW_COLOR, colorImage8U);
	cvReleaseImage(&colorImage8U);
	free(rgbimage);

	char c = waitKey( 1 );
	if(c == 'q') g_context.quit();
}

/*----------------------------------------------------------------------------*/
// New depth sample event handler
void onNewDepthSample(DepthNode node, DepthNode::NewSampleReceivedData data)
{
    int32_t w, h;
    FrameFormat_toResolution(data.captureConfiguration.frameFormat,&w,&h);
    
	if ( data.depthMap != nullptr ) {
		Mat mat( h, w, CV_16SC1, (void*)(const int16_t*)data.depthMap );
		imshow( WINDOW_DEPTH, mat );
		waitKey( 1 ); // ł1ms҂قۂ
	}
}

/*----------------------------------------------------------------------------*/
void configureAudioNode()
{
    g_anode.newSampleReceivedEvent().connect(&onNewAudioSample);

    AudioNode::Configuration config = g_anode.getConfiguration();
    config.sampleRate = 44100;

    try 
    {
        g_context.requestControl(g_anode,0);

        g_anode.setConfiguration(config);
        
        g_anode.setInputMixerLevel(0.5f);
    }
    catch (ArgumentException& e)
    {
        printf("Argument Exception: %s\n",e.what());
    }
    catch (UnauthorizedAccessException& e)
    {
        printf("Unauthorized Access Exception: %s\n",e.what());
    }
    catch (ConfigurationException& e)
    {
        printf("Configuration Exception: %s\n",e.what());
    }
    catch (StreamingException& e)
    {
        printf("Streaming Exception: %s\n",e.what());
    }
    catch (TimeoutException&)
    {
        printf("TimeoutException\n");
    }
}

/*----------------------------------------------------------------------------*/
void configureDepthNode()
{
    g_dnode.newSampleReceivedEvent().connect(&onNewDepthSample);

    DepthNode::Configuration config = g_dnode.getConfiguration();
    config.frameFormat = FRAME_FORMAT_QVGA;
    config.framerate = 25;
    config.mode = DepthNode::CAMERA_MODE_CLOSE_MODE;
    config.saturation = true;

    g_dnode.setEnableVertices(true);

	g_dnode.setEnableDepthMap( true );	// enable

    try 
    {
        g_context.requestControl(g_dnode,0);

        g_dnode.setConfiguration(config);
    }
    catch (ArgumentException& e)
    {
        printf("Argument Exception: %s\n",e.what());
    }
    catch (UnauthorizedAccessException& e)
    {
        printf("Unauthorized Access Exception: %s\n",e.what());
    }
    catch (IOException& e)
    {
        printf("IO Exception: %s\n",e.what());
    }
    catch (InvalidOperationException& e)
    {
        printf("Invalid Operation Exception: %s\n",e.what());
    }
    catch (ConfigurationException& e)
    {
        printf("Configuration Exception: %s\n",e.what());
    }
    catch (StreamingException& e)
    {
        printf("Streaming Exception: %s\n",e.what());
    }
    catch (TimeoutException&)
    {
        printf("TimeoutException\n");
    }

}

/*----------------------------------------------------------------------------*/
void configureColorNode()
{
    // connect new color sample handler
    g_cnode.newSampleReceivedEvent().connect(&onNewColorSample);

    ColorNode::Configuration config = g_cnode.getConfiguration();
    config.frameFormat = FRAME_FORMAT_VGA;
	config.compression = COMPRESSION_TYPE_YUY2;
//	config.compression = COMPRESSION_TYPE_MJPEG;
    config.powerLineFrequency = POWER_LINE_FREQUENCY_50HZ;
    config.framerate = 25;

    g_cnode.setEnableColorMap(true);

    try 
    {
        g_context.requestControl(g_cnode,0);

        g_cnode.setConfiguration(config);
    }
    catch (ArgumentException& e)
    {
        printf("Argument Exception: %s\n",e.what());
    }
    catch (UnauthorizedAccessException& e)
    {
        printf("Unauthorized Access Exception: %s\n",e.what());
    }
    catch (IOException& e)
    {
        printf("IO Exception: %s\n",e.what());
    }
    catch (InvalidOperationException& e)
    {
        printf("Invalid Operation Exception: %s\n",e.what());
    }
    catch (ConfigurationException& e)
    {
        printf("Configuration Exception: %s\n",e.what());
    }
    catch (StreamingException& e)
    {
        printf("Streaming Exception: %s\n",e.what());
    }
    catch (TimeoutException&)
    {
        printf("TimeoutException\n");
    }
}

/*----------------------------------------------------------------------------*/
void configureNode(Node node)
{
    if ((node.is<DepthNode>())&&(!g_dnode.isSet()))
    {
        g_dnode = node.as<DepthNode>();
        configureDepthNode();
        g_context.registerNode(node);
    }

    if ((node.is<ColorNode>())&&(!g_cnode.isSet()))
    {
        g_cnode = node.as<ColorNode>();
        configureColorNode();
        g_context.registerNode(node);
    }

    if ((node.is<AudioNode>())&&(!g_anode.isSet()))
    {
        g_anode = node.as<AudioNode>();
        configureAudioNode();
        g_context.registerNode(node);
    }
}

/*----------------------------------------------------------------------------*/
void onNodeConnected(Device device, Device::NodeAddedData data)
{
    configureNode(data.node);
}

/*----------------------------------------------------------------------------*/
void onNodeDisconnected(Device device, Device::NodeRemovedData data)
{
    if (data.node.is<AudioNode>() && (data.node.as<AudioNode>() == g_anode))
        g_anode.unset();
    if (data.node.is<ColorNode>() && (data.node.as<ColorNode>() == g_cnode))
        g_cnode.unset();
    if (data.node.is<DepthNode>() && (data.node.as<DepthNode>() == g_dnode))
        g_dnode.unset();
    printf("Node disconnected\n");
}

/*----------------------------------------------------------------------------*/
void onDeviceConnected(Context context, Context::DeviceAddedData data)
{
    if (!g_bDeviceFound)
    {
        data.device.nodeAddedEvent().connect(&onNodeConnected);
        data.device.nodeRemovedEvent().connect(&onNodeDisconnected);
        g_bDeviceFound = true;
    }
}

/*----------------------------------------------------------------------------*/
void onDeviceDisconnected(Context context, Context::DeviceRemovedData data)
{
    g_bDeviceFound = false;
    printf("Device disconnected\n");
}

/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[])
{
	cvNamedWindow(WINDOW_COLOR, CV_WINDOW_AUTOSIZE);	// ColorEBhE

    g_context = Context::create("localhost");

    g_context.deviceAddedEvent().connect(&onDeviceConnected);
    g_context.deviceRemovedEvent().connect(&onDeviceDisconnected);

    // Get the list of currently connected devices
    vector<Device> da = g_context.getDevices();

    // We are only interested in the first device
    if (da.size() >= 1)
    {
        g_bDeviceFound = true;

        da[0].nodeAddedEvent().connect(&onNodeConnected);
        da[0].nodeRemovedEvent().connect(&onNodeDisconnected);

        vector<Node> na = da[0].getNodes();
        
        printf("Found %u nodes\n",na.size());
        
        for (int n = 0; n < (int)na.size();n++)
            configureNode(na[n]);
    }

    g_context.startNodes();

    g_context.run();

    g_context.stopNodes();

    if (g_cnode.isSet()) g_context.unregisterNode(g_cnode);
    if (g_dnode.isSet()) g_context.unregisterNode(g_dnode);
    if (g_anode.isSet()) g_context.unregisterNode(g_anode);

    if (g_pProjHelper)
        delete g_pProjHelper;

    return 0;
}
